<?php

/**
 * Packet.ai GPU Cloud — WHMCS Provisioning Module
 *
 * Provisions, suspends, unsuspends, and terminates GPU pods on the
 * Packet.ai platform via the tenant API. Designed for white-label
 * hosting providers who resell GPU compute through WHMCS.
 *
 * @see https://packet.ai
 */

if (!defined('WHMCS')) {
    die('This file cannot be accessed directly.');
}

require_once __DIR__ . '/lib/PacketaiApi.php';

use Packetai\PacketaiApi;

// ------------------------------------------------------------------
//  Module metadata
// ------------------------------------------------------------------

/**
 * Module metadata returned to WHMCS.
 *
 * @return array{DisplayName: string, APIVersion: string, RequiresServer: bool}
 */
function packetai_MetaData(): array
{
    return [
        'DisplayName'    => 'Packet.ai GPU Cloud',
        'APIVersion'     => '1.1',
        'RequiresServer' => true,
    ];
}

// ------------------------------------------------------------------
//  Configurable product options
// ------------------------------------------------------------------

/**
 * Product configuration options displayed in the WHMCS admin when
 * setting up a product that uses this module.
 *
 * Option 1 (configoption1): GPU type dropdown.
 * Option 2 (configoption2): Default SSH public key (optional).
 *
 * @return array<string, array<string, mixed>>
 */
function packetai_ConfigOptions(): array
{
    return [
        'gpu_type' => [
            'FriendlyName' => 'GPU Type',
            'Type'         => 'dropdown',
            'Options'      => 'rtx-pro-6000|b200|h200|h100',
            'Description'  => 'Select the GPU model to provision.',
            'Default'      => 'rtx-pro-6000',
        ],
        'ssh_key' => [
            'FriendlyName' => 'SSH Public Key',
            'Type'         => 'text',
            'Size'         => 80,
            'Description'  => 'Default SSH public key injected into new pods (optional).',
            'Default'      => '',
        ],
    ];
}

// ------------------------------------------------------------------
//  Server connection test
// ------------------------------------------------------------------

/**
 * Test the connection to the Packet.ai API using the configured
 * server credentials.
 *
 * @param array $params WHMCS server parameters.
 *
 * @return array{success: bool, error?: string}
 */
function packetai_TestConnection(array $params): array
{
    try {
        $api = _packetai_buildClient($params);
        $api->testConnection();

        return ['success' => true];
    } catch (\Exception $e) {
        return [
            'success' => false,
            'error'   => $e->getMessage(),
        ];
    }
}

// ------------------------------------------------------------------
//  Provisioning lifecycle
// ------------------------------------------------------------------

/**
 * Provision a new GPU pod for the customer.
 *
 * On success the pod ID is persisted in the service's custom fields
 * so that subsequent lifecycle actions can reference it.
 *
 * @param array $params WHMCS module parameters.
 *
 * @return string 'success' or an error message.
 */
function packetai_CreateAccount(array $params): string
{
    try {
        $api      = _packetai_buildClient($params);
        $gpuType  = $params['configoption1'] ?? 'rtx-pro-6000';
        $clientId = (string) ($params['clientsdetails']['userid'] ?? $params['userid'] ?? '');

        $options = [];

        // Attach the SSH key when configured.
        $sshKey = trim($params['configoption2'] ?? '');
        if ($sshKey !== '') {
            $options['ssh_key'] = $sshKey;
        }

        $pod   = $api->createPod($gpuType, $clientId, $options);
        $podId = $pod->id ?? '';

        if ($podId === '') {
            return 'Provisioning succeeded but the API did not return a pod ID.';
        }

        // Persist the pod ID in WHMCS service properties.
        $params['model']->serviceProperties->save(['Pod ID' => (string) $podId]);

        return 'success';
    } catch (\Exception $e) {
        return $e->getMessage();
    }
}

/**
 * Suspend an active GPU pod.
 *
 * @param array $params WHMCS module parameters.
 *
 * @return string 'success' or an error message.
 */
function packetai_SuspendAccount(array $params): string
{
    try {
        $api   = _packetai_buildClient($params);
        $podId = _packetai_getPodId($params);

        $api->suspendPod($podId);

        return 'success';
    } catch (\Exception $e) {
        return $e->getMessage();
    }
}

/**
 * Unsuspend (resume) a previously suspended GPU pod.
 *
 * @param array $params WHMCS module parameters.
 *
 * @return string 'success' or an error message.
 */
function packetai_UnsuspendAccount(array $params): string
{
    try {
        $api   = _packetai_buildClient($params);
        $podId = _packetai_getPodId($params);

        $api->unsuspendPod($podId);

        return 'success';
    } catch (\Exception $e) {
        return $e->getMessage();
    }
}

/**
 * Permanently terminate a GPU pod and release all resources.
 *
 * @param array $params WHMCS module parameters.
 *
 * @return string 'success' or an error message.
 */
function packetai_TerminateAccount(array $params): string
{
    try {
        $api   = _packetai_buildClient($params);
        $podId = _packetai_getPodId($params);

        $api->terminatePod($podId);

        return 'success';
    } catch (\Exception $e) {
        return $e->getMessage();
    }
}

// ------------------------------------------------------------------
//  Package change (upgrade / downgrade)
// ------------------------------------------------------------------

/**
 * Handle a package change by terminating the existing pod and
 * provisioning a new one with the updated GPU type.
 *
 * The new pod ID is saved back into the service's custom fields.
 *
 * @param array $params WHMCS module parameters.
 *
 * @return string 'success' or an error message.
 */
function packetai_ChangePackage(array $params): string
{
    try {
        $api   = _packetai_buildClient($params);
        $podId = _packetai_getPodId($params);

        // Terminate the existing pod.
        $api->terminatePod($podId);

        // Provision a replacement pod with the new GPU type.
        $gpuType  = $params['configoption1'] ?? 'rtx-pro-6000';
        $clientId = (string) ($params['clientsdetails']['userid'] ?? $params['userid'] ?? '');

        $options = [];

        $sshKey = trim($params['configoption2'] ?? '');
        if ($sshKey !== '') {
            $options['ssh_key'] = $sshKey;
        }

        $pod      = $api->createPod($gpuType, $clientId, $options);
        $newPodId = $pod->id ?? '';

        if ($newPodId === '') {
            return 'Upgrade succeeded but the API did not return a new pod ID.';
        }

        $params['model']->serviceProperties->save(['Pod ID' => (string) $newPodId]);

        return 'success';
    } catch (\Exception $e) {
        return $e->getMessage();
    }
}

// ------------------------------------------------------------------
//  Internal helpers (prefixed with underscore to avoid collisions)
// ------------------------------------------------------------------

/**
 * Build a PacketaiApi client from the WHMCS server parameters.
 *
 * @param array $params WHMCS module parameters containing server details.
 *
 * @return PacketaiApi Configured API client.
 *
 * @throws \InvalidArgumentException If the server hostname or access hash is missing.
 */
function _packetai_buildClient(array $params): PacketaiApi
{
    $hostname = trim($params['serverhostname'] ?? '');
    $apiKey   = trim($params['serveraccesshash'] ?? '');

    if ($hostname === '') {
        throw new \InvalidArgumentException('Server hostname is not configured.');
    }

    if ($apiKey === '') {
        throw new \InvalidArgumentException('API key (access hash) is not configured.');
    }

    // Ensure the hostname includes a scheme.
    if (strpos($hostname, 'https://') !== 0 && strpos($hostname, 'http://') !== 0) {
        $hostname = 'https://' . $hostname;
    }

    return new PacketaiApi($hostname, $apiKey);
}

/**
 * Retrieve the stored pod ID from the service's custom fields.
 *
 * @param array $params WHMCS module parameters.
 *
 * @return string The pod ID.
 *
 * @throws \RuntimeException If no pod ID is found.
 */
function _packetai_getPodId(array $params): string
{
    $podId = trim($params['customfields']['Pod ID'] ?? '');

    if ($podId === '') {
        throw new \RuntimeException('No Pod ID found for this service. The pod may not have been provisioned yet.');
    }

    return $podId;
}
