<?php

declare(strict_types=1);

namespace Packetai;

/**
 * Packet.ai REST API client for WHMCS integration.
 *
 * Wraps the Packet.ai tenant API so hosting providers can provision,
 * manage, and bill GPU pods through WHMCS.
 *
 * Authentication uses an API key sent in the X-API-Key header.
 * All responses are decoded from JSON; errors throw \RuntimeException.
 */
class PacketaiApi
{
    /** @var string Base URL of the Packet.ai API (no trailing slash). */
    private string $apiUrl;

    /** @var string Tenant API key for authentication. */
    private string $apiKey;

    /** @var int HTTP request timeout in seconds. */
    private int $timeout = 30;

    /** @var string User-Agent string sent with every request. */
    private string $userAgent = 'PacketaiWHMCS/1.0';

    /**
     * Create a new API client.
     *
     * @param string $apiUrl Base URL of the Packet.ai API (e.g. "https://app.packet.ai").
     * @param string $apiKey Tenant API key.
     *
     * @throws \InvalidArgumentException If the URL or key is empty.
     */
    public function __construct(string $apiUrl, string $apiKey)
    {
        $apiUrl = rtrim($apiUrl, '/');

        if ($apiUrl === '') {
            throw new \InvalidArgumentException('API URL must not be empty.');
        }

        if ($apiKey === '') {
            throw new \InvalidArgumentException('API key must not be empty.');
        }

        $this->apiUrl = $apiUrl;
        $this->apiKey = $apiKey;
    }

    // ------------------------------------------------------------------
    //  Public API methods
    // ------------------------------------------------------------------

    /**
     * Test connectivity by pinging the API config endpoint.
     *
     * @return bool True when the API responds with a 2xx status.
     *
     * @throws \RuntimeException On network or server errors.
     */
    public function testConnection(): bool
    {
        $this->request('GET', '/api/widget/config');

        return true;
    }

    /**
     * List available GPU types and their pricing.
     *
     * @return array Decoded JSON array of GPU objects.
     *
     * @throws \RuntimeException On network or server errors.
     */
    public function listGpus(): array
    {
        $response = $this->request('GET', '/api/widget/pricing');

        return (array) $response;
    }

    /**
     * Provision a new GPU pod.
     *
     * @param string $gpuType  GPU model identifier (e.g. "rtx_pro_6000").
     * @param string $customerId  Tenant customer ID to associate with the pod.
     * @param array  $options  Optional parameters forwarded to the API
     *                         (e.g. region, image, ssh_key).
     *
     * @return object Decoded JSON object representing the created pod.
     *
     * @throws \InvalidArgumentException If required parameters are empty.
     * @throws \RuntimeException On network or server errors.
     */
    public function createPod(string $gpuType, string $customerId, array $options = []): object
    {
        if ($gpuType === '') {
            throw new \InvalidArgumentException('GPU type must not be empty.');
        }

        if ($customerId === '') {
            throw new \InvalidArgumentException('Customer ID must not be empty.');
        }

        $payload = array_merge($options, [
            'gpu_type'    => $gpuType,
            'customer_id' => $customerId,
        ]);

        return $this->request('POST', '/api/v1/tenant/pods', $payload);
    }

    /**
     * Retrieve details for a single pod.
     *
     * @param string $podId Pod identifier.
     *
     * @return object Decoded JSON object for the pod.
     *
     * @throws \InvalidArgumentException If the pod ID is empty.
     * @throws \RuntimeException On network or server errors.
     */
    public function getPod(string $podId): object
    {
        $this->validateId($podId, 'Pod ID');

        return $this->request('GET', '/api/v1/tenant/pods/' . urlencode($podId));
    }

    /**
     * Suspend a running pod.
     *
     * @param string $podId Pod identifier.
     *
     * @return bool True on success.
     *
     * @throws \InvalidArgumentException If the pod ID is empty.
     * @throws \RuntimeException On network or server errors.
     */
    public function suspendPod(string $podId): bool
    {
        $this->validateId($podId, 'Pod ID');

        $this->request('POST', '/api/v1/tenant/pods/' . urlencode($podId) . '/suspend');

        return true;
    }

    /**
     * Unsuspend (resume) a suspended pod.
     *
     * @param string $podId Pod identifier.
     *
     * @return bool True on success.
     *
     * @throws \InvalidArgumentException If the pod ID is empty.
     * @throws \RuntimeException On network or server errors.
     */
    public function unsuspendPod(string $podId): bool
    {
        $this->validateId($podId, 'Pod ID');

        $this->request('POST', '/api/v1/tenant/pods/' . urlencode($podId) . '/unsuspend');

        return true;
    }

    /**
     * Terminate (delete) a pod permanently.
     *
     * @param string $podId Pod identifier.
     *
     * @return bool True on success.
     *
     * @throws \InvalidArgumentException If the pod ID is empty.
     * @throws \RuntimeException On network or server errors.
     */
    public function terminatePod(string $podId): bool
    {
        $this->validateId($podId, 'Pod ID');

        $this->request('DELETE', '/api/v1/tenant/pods/' . urlencode($podId));

        return true;
    }

    /**
     * Retrieve usage data for billing.
     *
     * @param string|null $since Optional ISO-8601 date to filter usage from
     *                           (e.g. "2026-01-01T00:00:00Z").
     *
     * @return array Decoded JSON array of usage records.
     *
     * @throws \RuntimeException On network or server errors.
     */
    public function getUsage(?string $since = null): array
    {
        $path = '/api/v1/tenant/usage';

        if ($since !== null && $since !== '') {
            $path .= '?since=' . urlencode($since);
        }

        $response = $this->request('GET', $path);

        return (array) $response;
    }

    // ------------------------------------------------------------------
    //  Internal helpers
    // ------------------------------------------------------------------

    /**
     * Execute an HTTP request against the Packet.ai API.
     *
     * @param string     $method  HTTP method (GET, POST, DELETE, etc.).
     * @param string     $path    URI path including leading slash.
     * @param array|null $body    Optional JSON body for POST/PUT requests.
     *
     * @return mixed Decoded JSON response (object, array, or scalar).
     *
     * @throws \RuntimeException On cURL errors, non-2xx responses, or JSON decode failures.
     */
    private function request(string $method, string $path, ?array $body = null)
    {
        $url = $this->apiUrl . $path;

        $ch = curl_init();

        if ($ch === false) {
            throw new \RuntimeException('Failed to initialise cURL.');
        }

        try {
            $headers = [
                'X-API-Key: ' . $this->apiKey,
                'Accept: application/json',
                'User-Agent: ' . $this->userAgent,
            ];

            curl_setopt_array($ch, [
                CURLOPT_URL            => $url,
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_TIMEOUT        => $this->timeout,
                CURLOPT_CONNECTTIMEOUT => 10,
                CURLOPT_FOLLOWLOCATION => true,
                CURLOPT_MAXREDIRS      => 3,
                CURLOPT_CUSTOMREQUEST  => $method,
            ]);

            if ($body !== null) {
                $json = json_encode($body, JSON_THROW_ON_ERROR);
                $headers[] = 'Content-Type: application/json';
                $headers[] = 'Content-Length: ' . strlen($json);
                curl_setopt($ch, CURLOPT_POSTFIELDS, $json);
            }

            curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);

            $response   = curl_exec($ch);
            $curlErrno  = curl_errno($ch);
            $curlError  = curl_error($ch);
            $httpCode   = (int) curl_getinfo($ch, CURLINFO_HTTP_CODE);
        } finally {
            curl_close($ch);
        }

        // Handle cURL-level errors (DNS, timeout, connection refused, etc.)
        if ($curlErrno !== 0) {
            throw new \RuntimeException(
                sprintf('Packet.ai API request failed: [cURL %d] %s', $curlErrno, $curlError)
            );
        }

        // Decode JSON body (may be empty for 204 No Content)
        $decoded = null;

        if (is_string($response) && $response !== '') {
            $decoded = json_decode($response, false);

            if (json_last_error() !== JSON_ERROR_NONE) {
                throw new \RuntimeException(
                    sprintf(
                        'Packet.ai API returned invalid JSON (HTTP %d): %s',
                        $httpCode,
                        json_last_error_msg()
                    )
                );
            }
        }

        // Treat anything outside 2xx as an error
        if ($httpCode < 200 || $httpCode >= 300) {
            $message = 'Packet.ai API error';

            if (is_object($decoded) && isset($decoded->message)) {
                $message = (string) $decoded->message;
            } elseif (is_object($decoded) && isset($decoded->error)) {
                $message = (string) $decoded->error;
            }

            throw new \RuntimeException(
                sprintf('Packet.ai API returned HTTP %d: %s', $httpCode, $message)
            );
        }

        return $decoded;
    }

    /**
     * Validate that a string identifier is not empty.
     *
     * @param string $value The value to check.
     * @param string $label Human-readable label for the error message.
     *
     * @throws \InvalidArgumentException If the value is empty.
     */
    private function validateId(string $value, string $label): void
    {
        if ($value === '') {
            throw new \InvalidArgumentException($label . ' must not be empty.');
        }
    }
}
