<?php

/**
 * Packet.ai GPU Cloud — WHMCS Hooks
 *
 * Provides:
 *  1. DailyCronJob hook    — Collects usage data for all active Packet.ai services.
 *  2. AfterModuleCreate hook — Logs pod creation to the WHMCS activity log.
 *  3. packetai_ClientArea()  — Returns template and variables for the client area.
 *
 * @see https://packet.ai
 */

if (!defined('WHMCS')) {
    die('This file cannot be accessed directly.');
}

use WHMCS\Database\Capsule;

require_once __DIR__ . '/lib/PacketaiApi.php';

use Packetai\PacketaiApi;

// ------------------------------------------------------------------
//  Client Area function
// ------------------------------------------------------------------

/**
 * Render the client area page for a Packet.ai GPU pod service.
 *
 * This function is called by WHMCS when a customer views their
 * service details. It fetches the pod status and usage data from
 * the API, then returns the Smarty template path and variables.
 *
 * @param array $params WHMCS module parameters.
 *
 * @return array{templateFile: string, vars: array}
 */
function packetai_ClientArea(array $params): array
{
    $pod   = null;
    $usage = [];

    try {
        $api   = _packetai_buildClientFromParams($params);
        $podId = _packetai_extractPodId($params);

        if ($podId !== '') {
            // Fetch current pod status.
            $pod = $api->getPod($podId);

            // Fetch usage data for the current billing period.
            $billingStart = $params['model']->nextduedate ?? null;
            if ($billingStart) {
                // Go back one billing cycle to get the period start.
                $periodStart = date('Y-m-01\T00:00:00\Z', strtotime($billingStart . ' -1 month'));
            } else {
                // Fallback: start of the current month.
                $periodStart = date('Y-m-01\T00:00:00\Z');
            }

            $usageRecords = $api->getUsage($periodStart);
            $usage = _packetai_aggregateUsage($usageRecords, $podId);
        }
    } catch (\Exception $e) {
        // Silently degrade — the template handles missing $pod gracefully.
        logActivity(sprintf(
            'Packet.ai: Failed to load client area data for service #%d: %s',
            $params['serviceid'] ?? 0,
            $e->getMessage()
        ));
    }

    return [
        'templateFile' => 'clientarea',
        'vars'         => [
            'pod'   => $pod,
            'usage' => $usage,
        ],
    ];
}

// ------------------------------------------------------------------
//  DailyCronJob hook — Collect usage data for billing
// ------------------------------------------------------------------

add_hook('DailyCronJob', 0, function () {
    try {
        // Fetch all active services that use the packetai module.
        $services = Capsule::table('tblhosting')
            ->join('tblservers', 'tblhosting.server', '=', 'tblservers.id')
            ->where('tblservers.type', '=', 'packetai')
            ->where('tblhosting.domainstatus', '=', 'Active')
            ->select([
                'tblhosting.id as service_id',
                'tblhosting.nextduedate',
                'tblservers.hostname',
                'tblservers.accesshash',
            ])
            ->get();

        if ($services->isEmpty()) {
            return;
        }

        // Group services by server to reuse API clients.
        $grouped = [];
        foreach ($services as $service) {
            $key = $service->hostname . '|' . $service->accesshash;
            $grouped[$key]['server']     = $service;
            $grouped[$key]['services'][] = $service;
        }

        foreach ($grouped as $group) {
            $server = $group['server'];

            try {
                $hostname = trim($server->hostname);
                if (strpos($hostname, 'https://') !== 0 && strpos($hostname, 'http://') !== 0) {
                    $hostname = 'https://' . $hostname;
                }

                $api = new PacketaiApi($hostname, trim($server->accesshash));

                // Fetch usage for the current month.
                $since      = date('Y-m-01\T00:00:00\Z');
                $usageData  = $api->getUsage($since);

                // Index usage records by pod ID for quick lookup.
                $usageByPod = [];
                foreach ($usageData as $record) {
                    $record = (object) $record;
                    if (isset($record->pod_id)) {
                        $usageByPod[$record->pod_id] = $record;
                    }
                }

                foreach ($group['services'] as $service) {
                    try {
                        // Retrieve the pod ID from custom fields.
                        $podId = _packetai_getPodIdForService((int) $service->service_id);

                        if ($podId === '') {
                            continue;
                        }

                        $podUsage = $usageByPod[$podId] ?? null;

                        if ($podUsage === null) {
                            continue;
                        }

                        // Store the usage data as JSON in the overagedata field.
                        $overageData = json_encode([
                            'hours'              => $podUsage->hours ?? 0,
                            'estimated_cost'     => $podUsage->estimated_cost ?? 0,
                            'gpu_utilization'    => $podUsage->gpu_utilization ?? null,
                            'memory_utilization' => $podUsage->memory_utilization ?? null,
                            'collected_at'       => date('c'),
                        ], JSON_THROW_ON_ERROR);

                        Capsule::table('tblhosting')
                            ->where('id', $service->service_id)
                            ->update(['overagedata' => $overageData]);
                    } catch (\Exception $e) {
                        logActivity(sprintf(
                            'Packet.ai: Usage collection failed for service #%d: %s',
                            $service->service_id,
                            $e->getMessage()
                        ));
                    }
                }
            } catch (\Exception $e) {
                logActivity(sprintf(
                    'Packet.ai: Usage collection failed for server %s: %s',
                    $server->hostname,
                    $e->getMessage()
                ));
            }
        }

        logActivity('Packet.ai: Daily usage collection completed successfully.');
    } catch (\Exception $e) {
        logActivity('Packet.ai: Daily usage collection failed: ' . $e->getMessage());
    }
});

// ------------------------------------------------------------------
//  AfterModuleCreate hook — Log pod creation
// ------------------------------------------------------------------

add_hook('AfterModuleCreate', 0, function (array $params) {
    if (($params['server']['type'] ?? '') !== 'packetai') {
        return;
    }

    $serviceId  = $params['params']['serviceid'] ?? 0;
    $clientId   = $params['params']['userid'] ?? 0;
    $gpuType    = $params['params']['configoption1'] ?? 'unknown';

    logActivity(sprintf(
        'Packet.ai: GPU pod provisioned — Service #%d, Client #%d, GPU: %s',
        $serviceId,
        $clientId,
        $gpuType
    ));
});

// ------------------------------------------------------------------
//  Internal helpers
// ------------------------------------------------------------------

/**
 * Build an API client from WHMCS module parameters.
 *
 * This mirrors _packetai_buildClient() from the main module but is
 * scoped to the hooks file to avoid relying on an include of the
 * main module file (which WHMCS may not load for hook execution).
 *
 * @param array $params WHMCS module parameters.
 *
 * @return PacketaiApi
 *
 * @throws \InvalidArgumentException If server details are missing.
 */
function _packetai_buildClientFromParams(array $params): PacketaiApi
{
    $hostname = trim($params['serverhostname'] ?? '');
    $apiKey   = trim($params['serveraccesshash'] ?? '');

    if ($hostname === '') {
        throw new \InvalidArgumentException('Server hostname is not configured.');
    }

    if ($apiKey === '') {
        throw new \InvalidArgumentException('API key (access hash) is not configured.');
    }

    if (strpos($hostname, 'https://') !== 0 && strpos($hostname, 'http://') !== 0) {
        $hostname = 'https://' . $hostname;
    }

    return new PacketaiApi($hostname, $apiKey);
}

/**
 * Extract the pod ID from the service custom fields.
 *
 * @param array $params WHMCS module parameters.
 *
 * @return string Pod ID or empty string if not found.
 */
function _packetai_extractPodId(array $params): string
{
    return trim($params['customfields']['Pod ID'] ?? '');
}

/**
 * Retrieve the pod ID for a service by querying the custom fields table.
 *
 * Used by the DailyCronJob hook where full module params are not available.
 *
 * @param int $serviceId WHMCS hosting service ID.
 *
 * @return string Pod ID or empty string if not found.
 */
function _packetai_getPodIdForService(int $serviceId): string
{
    $field = Capsule::table('tblcustomfieldsvalues')
        ->join('tblcustomfields', 'tblcustomfieldsvalues.fieldid', '=', 'tblcustomfields.id')
        ->where('tblcustomfields.fieldname', '=', 'Pod ID')
        ->where('tblcustomfieldsvalues.relid', '=', $serviceId)
        ->value('tblcustomfieldsvalues.value');

    return trim((string) $field);
}

/**
 * Aggregate raw usage records into a summary for a specific pod.
 *
 * @param array  $usageRecords Raw usage records from the API.
 * @param string $podId        Pod ID to filter for.
 *
 * @return array Aggregated usage summary.
 */
function _packetai_aggregateUsage(array $usageRecords, string $podId): array
{
    $totalHours       = 0.0;
    $estimatedCost    = 0.0;
    $gpuUtilization   = null;
    $memUtilization   = null;
    $utilizationCount = 0;
    $periodStart      = null;
    $periodEnd        = null;

    foreach ($usageRecords as $record) {
        $record = (object) $record;

        // Filter to the specific pod if the API returns records for all pods.
        if (isset($record->pod_id) && $record->pod_id !== $podId) {
            continue;
        }

        $totalHours    += (float) ($record->hours ?? 0);
        $estimatedCost += (float) ($record->estimated_cost ?? 0);

        if (isset($record->gpu_utilization)) {
            $gpuUtilization = ($gpuUtilization ?? 0) + (float) $record->gpu_utilization;
            $utilizationCount++;
        }

        if (isset($record->memory_utilization)) {
            $memUtilization = ($memUtilization ?? 0) + (float) $record->memory_utilization;
        }

        // Track the date range.
        if (isset($record->date)) {
            if ($periodStart === null || $record->date < $periodStart) {
                $periodStart = $record->date;
            }
            if ($periodEnd === null || $record->date > $periodEnd) {
                $periodEnd = $record->date;
            }
        }
    }

    // Average the utilization values.
    if ($utilizationCount > 0) {
        $gpuUtilization /= $utilizationCount;
        if ($memUtilization !== null) {
            $memUtilization /= $utilizationCount;
        }
    }

    return [
        'hours'              => $totalHours,
        'estimated_cost'     => $estimatedCost,
        'gpu_utilization'    => $gpuUtilization,
        'memory_utilization' => $memUtilization,
        'period_start'       => $periodStart,
        'period_end'         => $periodEnd,
    ];
}
